<?php
/**
 * Minimal QBWC SOAP Server for OpenCart
 * - Uses a static WSDL at wsdl/QBWebConnectorSvc.wsdl
 * - Implements the required QBWC methods
 * - Currently returns "no work" so you can complete a successful handshake.
 *   (You can add Invoice export later — placeholders are marked TODO.)
 *
 * Common pitfalls solved:
 * - No stray whitespace before/after PHP tags (would break SOAP)
 * - Valid WSDL with <definitions> (fixes "Couldn't find <definitions>")
 * - Robust error logging into ./logs/qbwc.log
 */

// 0) Absolute requirement: do NOT emit anything before SOAP replies
declare(strict_types=1);
ini_set('display_errors', '0');
error_reporting(E_ALL & ~E_NOTICE);
ob_start();

require_once __DIR__ . '/config.php';


// === Helpers for qbXML state machine ===
function qbwc_state_path(): string { return LOG_DIR . '/work_state.json'; }
function qbwc_queue_path(): string { return LOG_DIR . '/next_request.xml'; }

function qbwc_state_read(): array {
    if (!file_exists(qbwc_state_path())) return [];
    $data = json_decode((string)@file_get_contents(qbwc_state_path()), true);
    return is_array($data) ? $data : [];
}
function qbwc_state_write(array $s): void {
    @file_put_contents(qbwc_state_path(), json_encode($s, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES));
}
function qbwc_enqueue(string $qbxml): void { @file_put_contents(qbwc_queue_path(), $qbxml); }
function qbwc_dequeue(): ?string {
    $p = qbwc_queue_path();
    if (!file_exists($p)) return null;
    $q = (string)@file_get_contents($p);
    @unlink($p);
    return $q ?: null;
}
function qbxml_wrap(string $body, string $ver='13.0'): string {
    return '<?xml version="1.0" encoding="utf-8"?>'
         . '<?qbxml version="'.$ver.'"?>'
         . '<QBXML><QBXMLMsgsRq onError="stopOnError">'
         . $body
         . '</QBXMLMsgsRq></QBXML>';
}
// === End helpers ===


// 1) Basic IP allow-list (optional)
if (defined('ALLOWED_IPS')) {
    $ok = false;
    $allowed = array_map('trim', explode(',', ALLOWED_IPS));
    if (!empty($_SERVER['REMOTE_ADDR']) && in_array($_SERVER['REMOTE_ADDR'], $allowed, true)) {
        $ok = true;
    }
    if (!$ok) {
        http_response_code(403);
        exit;
    }
}

// 2) Logging helper
function qbwc_log(string $msg): void {
    if (!is_dir(LOG_DIR)) { @mkdir(LOG_DIR, 0775, true); }
    $line = '[' . gmdate('Y-m-d H:i:s') . ' UTC] ' . $msg . PHP_EOL;
    @file_put_contents(LOG_DIR . '/qbwc.log', $line, FILE_APPEND);
}

// 3) PDO connection helper (you can use it later for order export)
function oc_pdo(): PDO {
    static $pdo = null;
    if ($pdo instanceof PDO) return $pdo;
    $dsn = sprintf('mysql:host=%s;port=%d;dbname=%s;charset=utf8mb4', OC_DB_HOST, OC_DB_PORT, OC_DB_NAME);
    $pdo = new PDO($dsn, OC_DB_USER, OC_DB_PASS, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
    return $pdo;
}

// 4) The service class used by SoapServer
class QBWCService {
    // Informational
    public function serverVersion(): string {
        qbwc_log('serverVersion()');
        return '2.0'; // anything is fine
    }
    public function clientVersion(string $productVersion): string {
        qbwc_log('clientVersion(' . $productVersion . ')');
        return ''; // '' means allow
    }

    // Login
    public function authenticate(string $strUserName, string $strPassword): array {
        qbwc_log("authenticate(user={$strUserName})");
        if ($strUserName === QBWC_USERNAME && $strPassword === QBWC_PASSWORD) {
            // Generate a session ticket
            $ticket = bin2hex(random_bytes(16));
            // Return: [ticket, company file, qbXML versions, minVersion]
            return [$ticket, COMPANY_FILE_PATH, '']; // third/fourth elements optional
        }
        // 'nvu' (not valid user) tells QBWC to stop
        return ['nvu', ''];
    }

    // The work queue — return qbXML to process OR empty string for "no work"
    public function sendRequestXML(string $ticket, string $strHCPResponse, string $strCompanyFileName,
                               string $qbXMLCountry, int $qbXMLMajorVers, int $qbXMLMinorVers): string {
    qbwc_log("sendRequestXML(ticket={$ticket}, file={$strCompanyFileName})");

    // Serve any queued one-shot qbXML first
    if ($queued = qbwc_dequeue()) {
        qbwc_log('sendRequestXML: serving queued qbXML');
        return $queued;
    }

    $s = qbwc_state_read();
    $step = $s['step'] ?? 'CHECK_ITEM';

    if ($step === 'CHECK_ITEM') {
        $s['step'] = 'ADD_ITEM_IF_NEEDED';
        qbwc_state_write($s);
        return qbxml_wrap('
            <ItemQueryRq requestID="1">
              <FullName>QBWC_TEST_ITEM</FullName>
            </ItemQueryRq>
        ');
    }

    if ($step === 'ADD_ITEM_IF_NEEDED') {
        return '';
    }

    if ($step === 'CHECK_CUSTOMER') {
        $s['step'] = 'ADD_CUSTOMER_IF_NEEDED';
        qbwc_state_write($s);
        return qbxml_wrap('
            <CustomerQueryRq requestID="2">
              <FullName>Web Customer</FullName>
            </CustomerQueryRq>
        ');
    }

    if ($step === 'ADD_CUSTOMER_IF_NEEDED') {
        return '';
    }

    if ($step === 'POST_SALESRECEIPT') {
        $s['step'] = 'DONE';
        qbwc_state_write($s);
        $today = date('Y-m-d');
        return qbxml_wrap('
            <SalesReceiptAddRq requestID="3">
              <SalesReceiptAdd>
                <CustomerRef><FullName>Web Customer</FullName></CustomerRef>
                <TxnDate>'.$today.'</TxnDate>
                <RefNumber>QBWC-TEST-'.time().'</RefNumber>
                <SalesReceiptLineAdd>
                  <ItemRef><FullName>QBWC_TEST_ITEM</FullName></ItemRef>
                  <Desc>Test post via QBWC</Desc>
                  <Quantity>1</Quantity>
                  <Rate>10.00</Rate>
                </SalesReceiptLineAdd>
              </SalesReceiptAdd>
            </SalesReceiptAddRq>
        ');
    }

    return '';
}

    // QBWC posts back the qbXML response; return a percentage 0-100
    public function receiveResponseXML(string $ticket, string $response, string $hresult, string $message): int {
    qbwc_log("receiveResponseXML(ticket={$ticket}, bytes=" . strlen($response) . ", hresult={$hresult}, message={$message})");

    $s = qbwc_state_read();
    $step = $s['step'] ?? 'CHECK_ITEM';

    $ok = (strpos($response, ' statusCode="0"') !== false);

    if ($step === 'ADD_ITEM_IF_NEEDED') {
        if ($ok && strpos($response, '<ItemRet>') !== false) {
            $s['step'] = 'CHECK_CUSTOMER';
        } else {
            $s['step'] = 'WAITING_ITEM_ADD';
            qbwc_enqueue(qbxml_wrap('
                <ItemNonInventoryAddRq requestID="1a">
                  <ItemNonInventoryAdd>
                    <Name>QBWC_TEST_ITEM</Name>
                    <SalesOrPurchase>
                      <Desc>QBWC Test Item</Desc>
                      <Price>10.00</Price>
                      <AccountRef><FullName>Sales</FullName></AccountRef>
                    </SalesOrPurchase>
                  </ItemNonInventoryAdd>
                </ItemNonInventoryAddRq>
            '));
        }
        qbwc_state_write($s);
        return 50;
    }

    if ($step === 'WAITING_ITEM_ADD') {
        $s['step'] = 'CHECK_CUSTOMER';
        qbwc_state_write($s);
        return 60;
    }

    if ($step === 'ADD_CUSTOMER_IF_NEEDED') {
        if ($ok && strpos($response, '<CustomerRet>') !== false) {
            $s['step'] = 'POST_SALESRECEIPT';
        } else {
            $s['step'] = 'WAITING_CUSTOMER_ADD';
            qbwc_enqueue(qbxml_wrap('
                <CustomerAddRq requestID="2a">
                  <CustomerAdd>
                    <Name>Web Customer</Name>
                  </CustomerAdd>
                </CustomerAddRq>
            '));
        }
        qbwc_state_write($s);
        return 75;
    }

    if ($step === 'WAITING_CUSTOMER_ADD') {
        $s['step'] = 'POST_SALESRECEIPT';
        qbwc_state_write($s);
        return 90;
    }

    if ($step === 'POST_SALESRECEIPT') {
        $s['step'] = 'DONE';
        qbwc_state_write($s);
        return 100;
    }

    return 100;
}

    public function connectionError(string $ticket, string $hresult, string $message): string {
        qbwc_log("connectionError(ticket={$ticket}, hresult={$hresult}, message={$message})");
        // Returning "done" or "cancel" tells QBWC to stop current session
        return 'done';
    }

    public function getLastError(string $ticket): string {
        qbwc_log("getLastError(ticket={$ticket})");
        return 'No error.';
    }

    public function closeConnection(string $ticket): string {
        qbwc_log("closeConnection(ticket={$ticket})");
        return 'OK';
    }
}

// 5) Create a WSDL-based SoapServer (this fixes your "Couldn't find <definitions>")
try {
    if (!is_readable(WSDL_ABS)) {
        qbwc_log('FATAL: WSDL not readable at ' . WSDL_ABS);
        throw new RuntimeException('WSDL missing');
    }
    $server = new SoapServer(WSDL_ABS, ['uri' => APP_URL]);
    $server->setClass(QBWCService::class);
    qbwc_log('SoapServer dispatch start');
    $server->handle();
    qbwc_log('SoapServer dispatch end');
} catch (Throwable $e) {
    qbwc_log('FATAL: ' . $e->getMessage());
    // On fatal, respond with HTTP 500 but *no* HTML (prevents "text/html; expected text/xml")
    http_response_code(500);
} finally {
    // Ensure no stray output survives
    if (ob_get_level() > 0) { ob_end_clean(); }
}
