<?php
/**
 * QuickBooks WC Desktop — Hand-rolled SOAP endpoint (no SoapServer)
 * Endpoint: qbwc_server2.php        |       WSDL: qbwc_server2.php?wsdl
 *
 * Notes in this build (v2):
 * - Robust QBXML extraction from SOAP (handles both raw and HTML-escaped <response>)
 * - 3-phase sync: items → customers → invoices
 * - InvoiceQueryRq simplified to avoid QB parser 0x80040400 (no date filter; include lines; cap MaxReturned)
 * - Better logging + last-request snapshots per phase
 * - CSV exports for items/customers/invoices (+ lines)
 */

declare(strict_types=1);
ini_set('display_errors','0');
ini_set('log_errors','1');
ini_set('error_log', __DIR__ . '/qbwc-php-error.log');
ini_set('zlib.output_compression','0');
ob_start();
// Keep server on UTC so date math is predictable
@date_default_timezone_set('UTC');

/* -------- tiny logger -------- */
const LOG_FILE = __DIR__ . '/qbwc-bridge.log';
function log_bridge(string $msg): void {
  @file_put_contents(LOG_FILE, '['.date('Y-m-d H:i:s')."] $msg\n", FILE_APPEND);
}

/* -------- helpers -------- */
function xml_header(): string { return '<?xml version="1.0" encoding="utf-8"?>'; }
function soap_env(string $bodyXml): string {
  return xml_header().
    '<soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/" '
  . 'xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" '
  . 'xmlns:xsd="http://www.w3.org/2001/XMLSchema" '
  . 'xmlns:soapenc="http://schemas.xmlsoap.org/soap/encoding/">'
  .   '<soap:Body soap:encodingStyle="http://schemas.xmlsoap.org/soap/encoding/">'
  .     $bodyXml
  .   '</soap:Body>'
  . '</soap:Envelope>';
}
function respond_soap(string $bodyXml): void {
  if (ob_get_length()) ob_clean();
  header('Content-Type: text/xml; charset=utf-8');
  echo soap_env($bodyXml);
  exit;
}
function respond_fault(string $faultString): void {
  if (ob_get_length()) ob_clean();
  header('Content-Type: text/xml; charset=utf-8');
  echo xml_header().
    '<SOAP-ENV:Envelope xmlns:SOAP-ENV="http://schemas.xmlsoap.org/soap/envelope/">'
  . '<SOAP-ENV:Body><SOAP-ENV:Fault>'
  .   '<faultcode>Server</faultcode>'
  .   '<faultstring>'.htmlspecialchars($faultString, ENT_QUOTES, 'UTF-8').'</faultstring>'
  . '</SOAP-ENV:Fault></SOAP-ENV:Body></SOAP-ENV:Envelope>';
  exit;
}

/* -------- state helpers -------- */
function state_dir(): string { $d = __DIR__ . '/state'; if (!is_dir($d)) @mkdir($d, 0775, true); return $d; }
function state_file(string $ticket): string {
  $safe = preg_replace('/[^A-Za-z0-9_\-\.]/','_', $ticket ?: 'default');
  return state_dir() . '/state-' . $safe . '.json';
}
function load_state(string $ticket): array {
  $f = state_file($ticket);
  if (is_file($f)) {
    $j = @json_decode((string)@file_get_contents($f), true);
    if (is_array($j)) return $j;
  }
  return ['phase' => 'items'];
}
function save_state(string $ticket, array $state): void { @file_put_contents(state_file($ticket), json_encode($state, JSON_UNESCAPED_SLASHES)); }
function clear_state(string $ticket): void { $f = state_file($ticket); if (is_file($f)) @unlink($f); }

/* -------- WSDL (for ?wsdl) -------- */
$WSDL_XML = <<<XML
<?xml version="1.0" encoding="utf-8"?>
<definitions name="QBWebConnectorSvc"
 targetNamespace="http://developer.intuit.com/"
 xmlns:tns="http://developer.intuit.com/"
 xmlns:soap="http://schemas.xmlsoap.org/wsdl/soap/"
 xmlns:xsd="http://www.w3.org/2001/XMLSchema"
 xmlns:soapenc="http://schemas.xmlsoap.org/soap/encoding/"
 xmlns:wsdl="http://schemas.xmlsoap.org/wsdl/">

  <types>
    <xsd:schema targetNamespace="http://developer.intuit.com/">
      <xsd:complexType name="ArrayOfString">
        <xsd:complexContent>
          <xsd:restriction base="soapenc:Array">
            <xsd:attribute ref="soapenc:arrayType" wsdl:arrayType="xsd:string[]"/>
          </xsd:restriction>
        </xsd:complexContent>
      </xsd:complexType>
    </xsd:schema>
  </types>

  <message name="authenticateRequest">
    <part name="strUserName" type="xsd:string"/>
    <part name="strPassword" type="xsd:string"/>
  </message>
  <message name="authenticateResponse"><part name="authenticateResult" type="tns:ArrayOfString"/></message>

  <message name="serverVersionRequest"/>
  <message name="serverVersionResponse"><part name="serverVersionResult" type="xsd:string"/></message>
  <message name="clientVersionRequest"><part name="strVersion" type="xsd:string"/></message>
  <message name="clientVersionResponse"><part name="clientVersionResult" type="xsd:string"/></message>

  <message name="sendRequestXMLRequest">
    <part name="ticket" type="xsd:string"/>
    <part name="strHCPResponse" type="xsd:string"/>
    <part name="strCompanyFileName" type="xsd:string"/>
    <part name="qbXMLCountry" type="xsd:string"/>
    <part name="qbXMLMajorVers" type="xsd:int"/>
    <part name="qbXMLMinorVers" type="xsd:int"/>
  </message>
  <message name="sendRequestXMLResponse"><part name="sendRequestXMLResult" type="xsd:string"/></message>

  <message name="receiveResponseXMLRequest">
    <part name="ticket" type="xsd:string"/>
    <part name="response" type="xsd:string"/>
    <part name="hresult" type="xsd:string"/>
    <part name="message" type="xsd:string"/>
  </message>
  <message name="receiveResponseXMLResponse"><part name="receiveResponseXMLResult" type="xsd:int"/></message>

  <message name="connectionErrorRequest">
    <part name="ticket" type="xsd:string"/>
    <part name="hresult" type="xsd:string"/>
    <part name="message" type="xsd:string"/>
  </message>
  <message name="connectionErrorResponse"><part name="connectionErrorResult" type="xsd:string"/></message>

  <message name="getLastErrorRequest"><part name="ticket" type="xsd:string"/></message>
  <message name="getLastErrorResponse"><part name="getLastErrorResult" type="xsd:string"/></message>

  <message name="closeConnectionRequest"><part name="ticket" type="xsd:string"/></message>
  <message name="closeConnectionResponse"><part name="closeConnectionResult" type="xsd:string"/></message>

  <portType name="QBWebConnectorSvcSoap">
    <operation name="authenticate"><input message="tns:authenticateRequest"/><output message="tns:authenticateResponse"/></operation>
    <operation name="serverVersion"><input message="tns:serverVersionRequest"/><output message="tns:serverVersionResponse"/></operation>
    <operation name="clientVersion"><input message="tns:clientVersionRequest"/><output message="tns:clientVersionResponse"/></operation>
    <operation name="sendRequestXML"><input message="tns:sendRequestXMLRequest"/><output message="tns:sendRequestXMLResponse"/></operation>
    <operation name="receiveResponseXML"><input message="tns:receiveResponseXMLRequest"/><output message="tns:receiveResponseXMLResponse"/></operation>
    <operation name="connectionError"><input message="tns:connectionErrorRequest"/><output message="tns:connectionErrorResponse"/></operation>
    <operation name="getLastError"><input message="tns:getLastErrorRequest"/><output message="tns:getLastErrorResponse"/></operation>
    <operation name="closeConnection"><input message="tns:closeConnectionRequest"/><output message="tns:closeConnectionResponse"/></operation>
  </portType>

  <binding name="QBWebConnectorSvcSoap" type="tns:QBWebConnectorSvcSoap">
    <soap:binding transport="http://schemas.xmlsoap.org/soap/http" style="rpc"/>
    <operation name="authenticate"><soap:operation soapAction="http://developer.intuit.com/authenticate"/><input><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></input><output><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></output></operation>
    <operation name="serverVersion"><soap:operation soapAction="http://developer.intuit.com/serverVersion"/><input><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></input><output><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></output></operation>
    <operation name="clientVersion"><soap:operation soapAction="http://developer.intuit.com/clientVersion"/><input><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></input><output><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></output></operation>
    <operation name="sendRequestXML"><soap:operation soapAction="http://developer.intuit.com/sendRequestXML"/><input><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></input><output><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></output></operation>
    <operation name="receiveResponseXML"><soap:operation soapAction="http://developer.intuit.com/receiveResponseXML"/><input><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></input><output><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></output></operation>
    <operation name="connectionError"><soap:operation soapAction="http://developer.intuit.com/connectionError"/><input><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></input><output><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></output></operation>
    <operation name="getLastError"><soap:operation soapAction="http://developer.intuit.com/getLastError"/><input><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></input><output><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></output></operation>
    <operation name="closeConnection"><soap:operation soapAction="http://developer.intuit.com/closeConnection"/><input><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></input><output><soap:body use="encoded" namespace="http://developer.intuit.com/" encodingStyle="http://schemas.xmlsoap.org/soap/encoding/"/></output></operation>
  </binding>

  <service name="QBWebConnectorSvc">
    <port name="QBWebConnectorSvcSoap" binding="tns:QBWebConnectorSvcSoap">
      <soap:address location="REPLACED_AT_RUNTIME"/>
    </port>
  </service>
</definitions>
XML;

/* -------- front controller -------- */
try {
  $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
  $self   = $scheme.'://'.$_SERVER['HTTP_HOST'].$_SERVER['SCRIPT_NAME'];
  $qs     = strtolower($_SERVER['QUERY_STRING'] ?? '');

  if ($qs === 'wsdl') {
    if (ob_get_length()) ob_clean();
    header('Content-Type: text/xml; charset=utf-8');
    echo str_replace('REPLACED_AT_RUNTIME', htmlspecialchars($self, ENT_QUOTES, 'UTF-8'), $WSDL_XML);
    exit;
  }

  $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
  if ($method === 'GET') {
    if (ob_get_length()) ob_clean();
    header('Content-Type: text/plain; charset=utf-8');
    echo "QBWC endpoint is alive. Use ?wsdl to view the WSDL.";
    exit;
  }

  $raw = file_get_contents('php://input') ?: '';
  $soapAction = trim((string)($_SERVER['HTTP_SOAPACTION'] ?? ''), "\"' ");
  $is = function(string $name) use ($soapAction, $raw): bool {
    if (stripos($soapAction, $name) !== false) return true;
    return (bool) preg_match('/<\s*(?:\w+:)?'.preg_quote($name,'/').'\b/i', $raw);
  };

  // --- serverVersion ---
  if ($is('serverVersion')) {
    respond_soap('<serverVersionResponse xmlns="http://developer.intuit.com/"><serverVersionResult xsi:type="xsd:string"></serverVersionResult></serverVersionResponse>');
  }

  // --- clientVersion ---
  if ($is('clientVersion')) {
    respond_soap('<clientVersionResponse xmlns="http://developer.intuit.com/"><clientVersionResult xsi:type="xsd:string"></clientVersionResult></clientVersionResponse>');
  }

  // --- authenticate ---
  if ($is('authenticate')) {
    // Strategy B: Force a specific company file (QB can auto-open it)
    $companyFile = 'C:\\Users\\Public\\Documents\\Intuit\\QuickBooks\\Company Files\\testintegration.qbw';
    $ticket = 'TOKEN123';
    log_bridge("AUTH REACHED user=[$ticket] companyFile=[$companyFile]");

    $body = '<authenticateResponse xmlns="http://developer.intuit.com/">'
          .   '<authenticateResult xsi:type="soapenc:Array" soapenc:arrayType="xsd:string[2]">'
          .     '<string xsi:type="xsd:string">'.htmlspecialchars($ticket, ENT_QUOTES, 'UTF-8').'</string>'
          .     '<string xsi:type="xsd:string">'.htmlspecialchars($companyFile, ENT_QUOTES, 'UTF-8').'</string>'
          .   '</authenticateResult>'
          . '</authenticateResponse>';
    respond_soap($body);
  }

  // --- sendRequestXML (phase-driven) ---
  if ($is('sendRequestXML')) {
    $maj = 16; $min = 0; // default to 16.0
    if (preg_match('/<qbXMLMajorVers[^>]*>(\d+)<\/qbXMLMajorVers>/i', $raw, $m)) $maj = (int) $m[1];
    if (preg_match('/<qbXMLMinorVers[^>]*>(\d+)<\/qbXMLMinorVers>/i', $raw, $m)) $min = (int) $m[1];

    $ticket = 'default';
    if (preg_match('/<ticket[^>]*>(.*?)<\/ticket>/is', $raw, $m)) {
      $ticket = trim(html_entity_decode($m[1], ENT_QUOTES | ENT_HTML5, 'UTF-8'));
    }

    $st = load_state($ticket);
    $phase = $st['phase'] ?? 'items';

    // Build ONE request per call
    $rq = '';
    if ($phase === 'items') {
      // Inventory items only (fast & safe)
      $rq = '<ItemInventoryQueryRq requestID="1">'
          .   '<MaxReturned>200</MaxReturned>'
          . '</ItemInventoryQueryRq>';
    } elseif ($phase === 'customers') {
      $rq = '<CustomerQueryRq requestID="2">'
          .   '<MaxReturned>200</MaxReturned>'
          . '</CustomerQueryRq>';
    } elseif ($phase === 'invoices') {
      // *** FIX: Keep this SIMPLE to avoid QB parser issues ***
      // No DateRangeFilter (some builds choke on order/combination). We just pull the latest few and include lines.
      $rq = '<InvoiceQueryRq requestID="3">'
          .   '<MaxReturned>10</MaxReturned>'
          .   '<IncludeLineItems>true</IncludeLineItems>'
          . '</InvoiceQueryRq>';
    } else {
      // Nothing more to send
      respond_soap('<sendRequestXMLResponse xmlns="http://developer.intuit.com/"><sendRequestXMLResult xsi:type="xsd:string"></sendRequestXMLResult></sendRequestXMLResponse>');
    }

    $qbxml = xml_header()
           . '<?qbxml version="'.$maj.'.'.$min.'"?>'
           . '<QBXML><QBXMLMsgsRq onError="stopOnError">'.$rq.'</QBXMLMsgsRq></QBXML>';

    // Snapshot for debugging
    $snap = __DIR__ . '/qbwc-last-request-' . $phase . '-' . date('Ymd_His') . '.qbxml';
    @file_put_contents($snap, $qbxml);
    log_bridge("sendRequestXML phase={$phase} v{$maj}.{$min}");

    respond_soap('<sendRequestXMLResponse xmlns="http://developer.intuit.com/"><sendRequestXMLResult xsi:type="xsd:string">'
      . htmlspecialchars($qbxml, ENT_NOQUOTES, 'UTF-8')
      . '</sendRequestXMLResult></sendRequestXMLResponse>');
  }

  // --- receiveResponseXML ---
  if ($is('receiveResponseXML')) {
    $ticket = 'default';
    if (preg_match('/<ticket[^>]*>(.*?)<\/ticket>/is', $raw, $m)) {
      $ticket = trim(html_entity_decode($m[1], ENT_QUOTES | ENT_HTML5, 'UTF-8'));
    }

    $stamp = date('Ymd_His');
    @file_put_contents(__DIR__ . "/qbwc-response-{$stamp}-soap.xml", $raw);

    // Pull pieces from SOAP
    $hresult = '';
    if (preg_match('/<hresult[^>]*>(.*?)<\/hresult>/is', $raw, $m)) $hresult = trim($m[1]);
    $message = '';
    if (preg_match('/<message[^>]*>(.*?)<\/message>/is', $raw, $m)) $message = trim($m[1]);

    $st = load_state($ticket); $phase = $st['phase'] ?? 'items';

    // Extract <response> payload (could be raw QBXML or HTML-escaped)
    $respPayload = '';
    if (preg_match('/<response[^>]*>([\s\S]*?)<\/response>/i', $raw, $m)) {
      $respPayload = trim($m[1]);
    }

    // Try to get real QBXML out of it
    $qbxml = '';
    if ($respPayload !== '') {
      // First try raw
      if (preg_match('/(<QBXML[\s\S]*?<\/QBXML>)/i', $respPayload, $mm)) {
        $qbxml = $mm[1];
      } else {
        // Then try HTML-escaped
        $decoded = html_entity_decode($respPayload, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        if (preg_match('/(<QBXML[\s\S]*?<\/QBXML>)/i', $decoded, $mm2)) {
          $qbxml = $mm2[1];
        }
      }
    }

    $progress = 100; $advanced = false; $note = '';

    if ($hresult !== '') {
      // QB couldn't parse or similar transport-level error
      $note = "HRESULT present ($hresult): $message";
      log_bridge("receiveResponseXML: $note");
      // Stay on current phase to retry next run
      $advanced = false; $progress = 100;
    } else if ($qbxml !== '') {
      // Parse QBXML
      $sx = @simplexml_load_string($qbxml);
      if ($sx && isset($sx->QBXMLMsgsRs)) {
        $msgs = $sx->QBXMLMsgsRs;
        $dir = __DIR__ . '/exports'; if (!is_dir($dir)) @mkdir($dir, 0775, true);

        if ($phase === 'items' && isset($msgs->ItemInventoryQueryRs)) {
          $fp = fopen($dir.'/products.csv','w');
          fputcsv($fp, ['ListID','Name','FullName','IsActive','SalesDesc','SalesPrice','IncomeAccount','COGSAccount','AssetAccount','QtyOnHand','TimeCreated','TimeModified']);
          foreach (($msgs->ItemInventoryQueryRs->ItemInventoryRet ?? []) as $r) {
            fputcsv($fp,[
              (string)$r->ListID,
              (string)$r->Name,
              (string)$r->FullName,
              (string)$r->IsActive,
              (string)$r->SalesDesc,
              (string)$r->SalesPrice,
              (string)($r->IncomeAccountRef->FullName ?? ''),
              (string)($r->COGSAccountRef->FullName ?? ''),
              (string)($r->AssetAccountRef->FullName ?? ''),
              (string)$r->QuantityOnHand,
              (string)$r->TimeCreated,
              (string)$r->TimeModified,
            ]);
          }
          fclose($fp);
          $st['phase'] = 'customers'; save_state($ticket, $st);
          $advanced = true; $progress = 33;
        } elseif ($phase === 'customers' && isset($msgs->CustomerQueryRs)) {
          $fp = fopen($dir.'/customers.csv','w');
          fputcsv($fp,['ListID','FullName','Name','CompanyName','FirstName','LastName','Bill_Addr1','Bill_Addr2','Bill_City','Bill_State','Bill_PostalCode','Bill_Country','Phone','Email','Balance','IsActive','TimeCreated','TimeModified']);
          foreach (($msgs->CustomerQueryRs->CustomerRet ?? []) as $c) {
            $b = $c->BillAddress ?? null;
            fputcsv($fp,[
              (string)$c->ListID,(string)$c->FullName,(string)$c->Name,(string)$c->CompanyName,
              (string)$c->FirstName,(string)$c->LastName,
              (string)($b->Addr1 ?? ''),(string)($b->Addr2 ?? ''),(string)($b->City ?? ''),(string)($b->State ?? ''),(string)($b->PostalCode ?? ''),(string)($b->Country ?? ''),
              (string)($c->Phone ?? ''),(string)($c->Email ?? ''),(string)($c->Balance ?? ''),(string)($c->IsActive ?? ''),
              (string)$c->TimeCreated,(string)$c->TimeModified,
            ]);
          }
          fclose($fp);
          $st['phase'] = 'invoices'; save_state($ticket, $st);
          $advanced = true; $progress = 66;
        } elseif ($phase === 'invoices' && isset($msgs->InvoiceQueryRs)) {
          // Invoices + lines
          $fpInv = fopen($dir.'/invoices.csv','w');
          fputcsv($fpInv,['TxnID','TxnDate','RefNumber','Customer','PONumber','ShipDate','Subtotal','SalesTaxTotal','AppliedAmount','BalanceRemaining','TotalAmount','TimeCreated','TimeModified']);
          $lines = [];
          foreach (($msgs->InvoiceQueryRs->InvoiceRet ?? []) as $inv) {
            fputcsv($fpInv,[
              (string)$inv->TxnID,(string)$inv->TxnDate,(string)$inv->RefNumber,(string)($inv->CustomerRef->FullName ?? ''),
              (string)$inv->PONumber,(string)$inv->ShipDate,(string)$inv->Subtotal,(string)$inv->SalesTaxTotal,
              (string)$inv->AppliedAmount,(string)$inv->BalanceRemaining,(string)$inv->TotalAmount,
              (string)$inv->TimeCreated,(string)$inv->TimeModified,
            ]);
            $txn = (string)$inv->TxnID;
            foreach (($inv->InvoiceLineRet ?? []) as $ln) {
              $lines[] = [$txn,(string)$ln->TxnLineID,(string)($ln->ItemRef->FullName ?? ''),(string)$ln->Desc,(string)$ln->Quantity,(string)($ln->Rate ?? $ln->RatePercent ?? ''),(string)$ln->Amount,'Line'];
            }
            foreach (($inv->InvoiceLineGroupRet ?? []) as $grp) {
              $g = (string)($grp->ItemGroupRef->FullName ?? '');
              foreach (($grp->InvoiceLineRet ?? []) as $ln) {
                $lines[] = [$txn,(string)$ln->TxnLineID,$g.' / '.(string)($ln->ItemRef->FullName ?? ''),(string)$ln->Desc,(string)$ln->Quantity,(string)($ln->Rate ?? $ln->RatePercent ?? ''),(string)$ln->Amount,'GroupMember'];
              }
            }
          }
          fclose($fpInv);
          if ($lines) { $fpL = fopen($dir.'/invoice_lines.csv','w'); fputcsv($fpL,['TxnID','TxnLineID','Item','Desc','Quantity','RateOrRatePercent','Amount','LineType']); foreach ($lines as $r) fputcsv($fpL,$r); fclose($fpL);}      
          // Done
          clear_state($ticket); $advanced = true; $progress = 100;
        }
      }

      log_bridge('receiveResponseXML: '.($qbxml ? 'parsed OK' : 'no QBXML extracted').'; advanced='.( $advanced ? 'yes' : 'no').'; progress='.$progress);
    } else {
      log_bridge('receiveResponseXML: no <QBXML> found in SOAP body');
    }

    // Return % complete
    respond_soap('<receiveResponseXMLResponse xmlns="http://developer.intuit.com/"><receiveResponseXMLResult xsi:type="xsd:int">'.$progress.'</receiveResponseXMLResult></receiveResponseXMLResponse>');
  }

  // --- connectionError ---
  if ($is('connectionError')) {
    log_bridge('connectionError');
    respond_soap('<connectionErrorResponse xmlns="http://developer.intuit.com/"><connectionErrorResult xsi:type="xsd:string">done</connectionErrorResult></connectionErrorResponse>');
  }

  // --- getLastError ---
  if ($is('getLastError')) {
    $last = '';
    respond_soap('<getLastErrorResponse xmlns="http://developer.intuit.com/"><getLastErrorResult xsi:type="xsd:string">'.htmlspecialchars($last, ENT_QUOTES, 'UTF-8').'</getLastErrorResult></getLastErrorResponse>');
  }

  // --- closeConnection ---
  if ($is('closeConnection')) {
    // End session + clear state for ticket
    $ticket = 'default';
    if (preg_match('/<ticket[^>]*>(.*?)<\/ticket>/is', $raw, $m)) $ticket = trim(html_entity_decode($m[1], ENT_QUOTES | ENT_HTML5, 'UTF-8'));
    clear_state($ticket);
    respond_soap('<closeConnectionResponse xmlns="http://developer.intuit.com/"><closeConnectionResult xsi:type="xsd:string">OK</closeConnectionResult></closeConnectionResponse>');
  }

  // Fallback
  log_bridge('Unknown SOAP call');
  respond_fault('Unknown SOAP method');

} catch (Throwable $e) {
  log_bridge('FATAL: '.$e->getMessage());
  respond_fault('Internal error');
}
